import tkinter as tk
from tkinter import ttk, messagebox, filedialog
import json
import base64
import os
from cryptography.fernet import Fernet
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
import csv
from datetime import datetime

class PasswordManager:
    def __init__(self, root):
        self.root = root
        self.root.title("パスワード管理ツール")
        self.root.geometry("1400x800")
        
        self.data_file = "passwords.encrypted"
        self.fernet = None
        self.passwords = []
        self.filtered_passwords = []
        self.current_salt = None
        
        self.types = ["銀行", "証券", "クレカ", "サービス", "キャッシュレス", 
                     "資産管理", "通信", "自動車保険", "基盤", "お薬手帳", 
                     "PC", "ブログ", "マイナンバー", "趣味", "ポイ活", 
                     "チケット", "ゲーム", "その他", "AI"]
        
        self.setup_login_screen()
    
    def derive_key(self, password: str, salt: bytes) -> bytes:
        """パスワードから暗号化キーを生成"""
        kdf = PBKDF2HMAC(
            algorithm=hashes.SHA256(),
            length=32,
            salt=salt,
            iterations=100000,
        )
        key = base64.urlsafe_b64encode(kdf.derive(password.encode()))
        return key
    
    def setup_login_screen(self):
        """ログイン画面のセットアップ"""
        self.login_frame = ttk.Frame(self.root, padding="50")
        self.login_frame.pack(expand=True)
        
        ttk.Label(self.login_frame, text="パスワード管理ツール", 
                 font=("Arial", 20, "bold")).pack(pady=20)
        
        ttk.Label(self.login_frame, text="マスターパスワード:").pack(pady=5)
        self.master_password_entry = ttk.Entry(self.login_frame, show="*", width=30)
        self.master_password_entry.pack(pady=5)
        self.master_password_entry.bind('<Return>', lambda e: self.login())
        
        btn_frame = ttk.Frame(self.login_frame)
        btn_frame.pack(pady=20)
        
        ttk.Button(btn_frame, text="ログイン", command=self.login).pack(side=tk.LEFT, padx=5)
        ttk.Button(btn_frame, text="新規作成", command=self.create_new).pack(side=tk.LEFT, padx=5)
        
        self.master_password_entry.focus()
    
    def login(self):
        """ログイン処理"""
        master_password = self.master_password_entry.get()
        if not master_password:
            messagebox.showwarning("警告", "マスターパスワードを入力してください")
            return
        
        if os.path.exists(self.data_file):
            try:
                with open(self.data_file, 'rb') as f:
                    salt = f.read(16)
                    encrypted_data = f.read()
                
                self.current_salt = salt
                key = self.derive_key(master_password, salt)
                self.fernet = Fernet(key)
                
                decrypted_data = self.fernet.decrypt(encrypted_data)
                self.passwords = json.loads(decrypted_data.decode())
                self.filtered_passwords = self.passwords.copy()
                
                self.login_frame.destroy()
                self.setup_main_screen()
            except Exception as e:
                messagebox.showerror("エラー", "マスターパスワードが正しくありません")
        else:
            messagebox.showerror("エラー", "データファイルが見つかりません\n新規作成してください")
    
    def create_new(self):
        """新規データファイル作成"""
        master_password = self.master_password_entry.get()
        if not master_password:
            messagebox.showwarning("警告", "マスターパスワードを入力してください")
            return
        
        if len(master_password) < 8:
            messagebox.showwarning("警告", "マスターパスワードは8文字以上にしてください")
            return
        
        if os.path.exists(self.data_file):
            if not messagebox.askyesno("確認", "既存のデータファイルを上書きしますか？"):
                return
        
        salt = os.urandom(16)
        self.current_salt = salt
        key = self.derive_key(master_password, salt)
        self.fernet = Fernet(key)
        self.passwords = []
        self.filtered_passwords = []
        
        self.save_data()
        
        messagebox.showinfo("成功", "新しいデータファイルを作成しました")
        self.login_frame.destroy()
        self.setup_main_screen()
    
    def save_data(self):
        """データの保存"""
        try:
            json_data = json.dumps(self.passwords, ensure_ascii=False, indent=2)
            encrypted_data = self.fernet.encrypt(json_data.encode())
            
            if os.path.exists(self.data_file):
                with open(self.data_file, 'rb') as f:
                    salt = f.read(16)
            else:
                salt = self.current_salt
            
            with open(self.data_file, 'wb') as f:
                f.write(salt)
                f.write(encrypted_data)
            return True
        except Exception as e:
            messagebox.showerror("エラー", f"保存に失敗しました: {str(e)}")
            return False
    
    def setup_main_screen(self):
        """メイン画面のセットアップ"""
        menubar = tk.Menu(self.root)
        self.root.config(menu=menubar)
        
        file_menu = tk.Menu(menubar, tearoff=0)
        menubar.add_cascade(label="ファイル", menu=file_menu)
        file_menu.add_command(label="CSVエクスポート", command=self.export_csv)
        file_menu.add_separator()
        file_menu.add_command(label="終了", command=self.root.quit)
        
        main_frame = ttk.Frame(self.root)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        # 画面全体の幅と高さを取得
        main_frame.update_idletasks()
        
        # 左側：入力・検索エリア（画面の55%）
        left_frame = ttk.Frame(main_frame)
        left_frame.place(relx=0, rely=0, relwidth=0.55, relheight=1)
        
        # 検索エリア
        search_frame = ttk.LabelFrame(left_frame, text="検索・フィルター", padding="10")
        search_frame.pack(fill=tk.X, pady=(0, 10))
        
        ttk.Label(search_frame, text="キーワード:").pack(anchor=tk.W)
        self.search_entry = ttk.Entry(search_frame)
        self.search_entry.pack(fill=tk.X, pady=(0, 5))
        self.search_entry.bind('<KeyRelease>', lambda e: self.filter_data())
        
        ttk.Label(search_frame, text="種類:").pack(anchor=tk.W)
        self.filter_type_combo = ttk.Combobox(search_frame, values=["すべて"] + self.types, state="readonly")
        self.filter_type_combo.set("すべて")
        self.filter_type_combo.pack(fill=tk.X)
        self.filter_type_combo.bind('<<ComboboxSelected>>', lambda e: self.filter_data())
        
        # 入力エリア
        input_frame = ttk.LabelFrame(left_frame, text="データ入力", padding="10")
        input_frame.pack(fill=tk.BOTH, expand=True)
        
        # スクロール可能なフレーム
        canvas = tk.Canvas(input_frame, height=400)
        scrollbar = ttk.Scrollbar(input_frame, orient="vertical", command=canvas.yview)
        scrollable_frame = ttk.Frame(canvas)
        
        scrollable_frame.bind(
            "<Configure>",
            lambda e: canvas.configure(scrollregion=canvas.bbox("all"))
        )
        
        canvas.create_window((0, 0), window=scrollable_frame, anchor="nw")
        canvas.configure(yscrollcommand=scrollbar.set)
        
        scrollable_frame.columnconfigure(1, weight=1)
        scrollable_frame.columnconfigure(2, weight=1)
        
        canvas.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        # 入力フィールド
        self.entries = {}
        row = 0
        
        # 固定フィールド
        fixed_fields = [
            ("大区分", "entry", False),
            ("小区分", "entry", False),
            ("種類", "combo", False),
            ("アカウント/ID名", "entry", True),
            ("パスワード", "entry", True),
            ("セキュリティコード", "entry", True),
            ("アドレス", "entry", True),
        ]
        
        for label, field_type, has_copy in fixed_fields:
            ttk.Label(scrollable_frame, text=f"{label}:").grid(row=row, column=0, sticky=tk.W, pady=3)
            
            if field_type == "combo":
                widget = ttk.Combobox(scrollable_frame, values=self.types, width=20 if has_copy else 25)
            else:
                widget = ttk.Entry(scrollable_frame, width=20 if has_copy else 25)
            
            widget.grid(row=row, column=1, columnspan=2 if not has_copy else 1, pady=3, sticky=tk.EW, padx=(0, 5))
            self.entries[label] = widget
            
            if has_copy:
                copy_btn = ttk.Button(scrollable_frame, text="コピー", width=6,
                                     command=lambda lbl=label: self.copy_to_clipboard(lbl))
                copy_btn.grid(row=row, column=3, pady=3, padx=(0, 5))
            
            row += 1
        
        # 任意項目（名称と値のペア）
        ttk.Label(scrollable_frame, text="--- 任意項目 ---", font=("Arial", 9, "bold")).grid(
            row=row, column=0, columnspan=4, pady=(10, 5), sticky=tk.W)
        row += 1
        
        self.custom_fields = []
        for i in range(1, 11):
            # 名称フィールド（幅を狭く）
            name_entry = ttk.Entry(scrollable_frame, width=50)
            name_entry.grid(row=row, column=0, pady=3, sticky=tk.EW, padx=(0, 3))
            
            # 値フィールド（マスクなし、幅広く）
            value_entry = ttk.Entry(scrollable_frame, width=50)
            value_entry.grid(row=row, column=1, columnspan=2, pady=3, sticky=tk.EW, padx=(0, 3))
            
            # コピーボタン
            copy_btn = ttk.Button(scrollable_frame, text="コピー", width=6,
                                 command=lambda ve=value_entry: self.copy_custom_field(ve))
            copy_btn.grid(row=row, column=3, pady=3, padx=(0, 5))
            
            self.custom_fields.append((name_entry, value_entry))
            row += 1
        
        # ボタンエリア（入力エリアの最下部に配置）
        btn_row_frame = ttk.Frame(scrollable_frame)
        btn_row_frame.grid(row=row, column=0, columnspan=4, pady=15, sticky=tk.EW)
        
        ttk.Button(btn_row_frame, text="追加", command=self.add_entry, width=10).pack(side=tk.LEFT, padx=5)
        ttk.Button(btn_row_frame, text="更新", command=self.update_entry, width=10).pack(side=tk.LEFT, padx=5)
        ttk.Button(btn_row_frame, text="クリア", command=self.clear_entries, width=10).pack(side=tk.LEFT, padx=5)
        
        # 右側：一覧表示エリア（画面の30%）
        right_frame = ttk.Frame(main_frame)
        right_frame.place(relx=0.58, rely=0, relwidth=0.40, relheight=1)
        
        list_frame = ttk.LabelFrame(right_frame, text="登録データ一覧", padding="10")
        list_frame.pack(fill=tk.BOTH, expand=True)
        
        tree_scroll = ttk.Scrollbar(list_frame)
        tree_scroll.pack(side=tk.RIGHT, fill=tk.Y)
        
        self.tree = ttk.Treeview(list_frame, yscrollcommand=tree_scroll.set, 
                                 columns=("大区分", "小区分", "種類", "アカウント"), 
                                 show="tree headings", selectmode="browse")
        self.tree.pack(fill=tk.BOTH, expand=True)
        tree_scroll.config(command=self.tree.yview)
        
        self.tree.heading("#0", text="No")
        self.tree.heading("大区分", text="大区分")
        self.tree.heading("小区分", text="小区分")
        self.tree.heading("種類", text="種類")
        self.tree.heading("アカウント", text="アカウント/ID")
        
        self.tree.column("#0", width=25)
        self.tree.column("大区分", width=100)
        self.tree.column("小区分", width=100)
        self.tree.column("種類", width=120)
        self.tree.column("アカウント", width=100)
        
        self.tree.bind('<<TreeviewSelect>>', self.on_select)
        
        ttk.Button(list_frame, text="選択項目を削除", command=self.delete_entry).pack(pady=5)
        
        self.refresh_tree()
    
    def copy_to_clipboard(self, field_label):
        """クリップボードにコピー"""
        try:
            value = self.entries[field_label].get()
            if value:
                self.root.clipboard_clear()
                self.root.clipboard_append(value)
                self.root.update()
            else:
                messagebox.showwarning("警告", "コピーする内容がありません")
        except Exception as e:
            messagebox.showerror("エラー", f"コピーに失敗しました: {str(e)}")
    
    def copy_custom_field(self, value_entry):
        """任意項目の値をコピー"""
        try:
            value = value_entry.get()
            if value:
                self.root.clipboard_clear()
                self.root.clipboard_append(value)
                self.root.update()
            else:
                messagebox.showwarning("警告", "コピーする内容がありません")
        except Exception as e:
            messagebox.showerror("エラー", f"コピーに失敗しました: {str(e)}")
    
    def filter_data(self):
        """データのフィルタリング"""
        keyword = self.search_entry.get().lower()
        type_filter = self.filter_type_combo.get()
        
        self.filtered_passwords = []
        for pwd in self.passwords:
            if type_filter != "すべて" and pwd.get("種類") != type_filter:
                continue
            
            if keyword:
                searchable = [
                    pwd.get("大区分", ""),
                    pwd.get("小区分", ""),
                    pwd.get("種類", ""),
                    pwd.get("アカウント/ID名", ""),
                    pwd.get("アドレス", "")
                ]
                if not any(keyword in str(s).lower() for s in searchable):
                    continue
            
            self.filtered_passwords.append(pwd)
        
        self.refresh_tree()
    
    def refresh_tree(self):
        """ツリービューの更新"""
        for item in self.tree.get_children():
            self.tree.delete(item)
        
        for i, pwd in enumerate(self.filtered_passwords, 1):
            self.tree.insert("", tk.END, text=str(i),
                           values=(pwd.get("大区分", ""),
                                  pwd.get("小区分", ""),
                                  pwd.get("種類", ""),
                                  pwd.get("アカウント/ID名", "")))
    
    def on_select(self, event):
        """ツリー選択時の処理"""
        selection = self.tree.selection()
        if not selection:
            return
        
        item = self.tree.item(selection[0])
        idx = int(item['text']) - 1
        
        if 0 <= idx < len(self.filtered_passwords):
            pwd = self.filtered_passwords[idx]
            
            # 固定フィールドの読み込み
            self.entries["大区分"].delete(0, tk.END)
            self.entries["大区分"].insert(0, pwd.get("大区分", ""))
            
            self.entries["小区分"].delete(0, tk.END)
            self.entries["小区分"].insert(0, pwd.get("小区分", ""))
            
            self.entries["種類"].set(pwd.get("種類", ""))
            
            self.entries["アカウント/ID名"].delete(0, tk.END)
            self.entries["アカウント/ID名"].insert(0, pwd.get("アカウント/ID名", ""))
            
            self.entries["パスワード"].delete(0, tk.END)
            self.entries["パスワード"].insert(0, pwd.get("パスワード", ""))
            
            self.entries["セキュリティコード"].delete(0, tk.END)
            self.entries["セキュリティコード"].insert(0, pwd.get("セキュリティコード", ""))
            
            self.entries["アドレス"].delete(0, tk.END)
            self.entries["アドレス"].insert(0, pwd.get("アドレス", ""))
            
            # 任意項目のクリア
            for name_entry, value_entry in self.custom_fields:
                name_entry.delete(0, tk.END)
                value_entry.delete(0, tk.END)
            
            # 任意項目の読み込み（カスタムフィールド）
            custom_data = pwd.get("custom_fields", {})
            idx = 0
            for field_name, field_value in custom_data.items():
                if idx < len(self.custom_fields):
                    self.custom_fields[idx][0].insert(0, field_name)
                    self.custom_fields[idx][1].insert(0, field_value)
                    idx += 1
    
    def add_entry(self):
        """エントリの追加"""
        new_entry = {
            "大区分": self.entries["大区分"].get(),
            "小区分": self.entries["小区分"].get(),
            "種類": self.entries["種類"].get(),
            "アカウント/ID名": self.entries["アカウント/ID名"].get(),
            "パスワード": self.entries["パスワード"].get(),
            "セキュリティコード": self.entries["セキュリティコード"].get(),
            "アドレス": self.entries["アドレス"].get(),
            "作成日時": datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        }
        
        # 任意項目の収集
        custom_fields = {}
        for name_entry, value_entry in self.custom_fields:
            field_name = name_entry.get().strip()
            field_value = value_entry.get()
            if field_name:  # 名称が入力されている場合のみ保存
                custom_fields[field_name] = field_value
        
        if custom_fields:
            new_entry["custom_fields"] = custom_fields
        
        self.passwords.append(new_entry)
        if self.save_data():
            messagebox.showinfo("成功", "データを追加しました")
            self.clear_entries()
            self.filter_data()
    
    def update_entry(self):
        """エントリの更新"""
        selection = self.tree.selection()
        if not selection:
            messagebox.showwarning("警告", "更新する項目を選択してください")
            return
        
        item = self.tree.item(selection[0])
        idx = int(item['text']) - 1
        
        if 0 <= idx < len(self.filtered_passwords):
            old_entry = self.filtered_passwords[idx]
            original_idx = self.passwords.index(old_entry)
            
            updated_entry = {
                "大区分": self.entries["大区分"].get(),
                "小区分": self.entries["小区分"].get(),
                "種類": self.entries["種類"].get(),
                "アカウント/ID名": self.entries["アカウント/ID名"].get(),
                "パスワード": self.entries["パスワード"].get(),
                "セキュリティコード": self.entries["セキュリティコード"].get(),
                "アドレス": self.entries["アドレス"].get(),
                "作成日時": old_entry.get("作成日時", ""),
                "更新日時": datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            }
            
            # 任意項目の収集
            custom_fields = {}
            for name_entry, value_entry in self.custom_fields:
                field_name = name_entry.get().strip()
                field_value = value_entry.get()
                if field_name:
                    custom_fields[field_name] = field_value
            
            if custom_fields:
                updated_entry["custom_fields"] = custom_fields
            
            self.passwords[original_idx] = updated_entry
            if self.save_data():
                messagebox.showinfo("成功", "データを更新しました")
                self.filter_data()
    
    def delete_entry(self):
        """エントリの削除"""
        selection = self.tree.selection()
        if not selection:
            messagebox.showwarning("警告", "削除する項目を選択してください")
            return
        
        if not messagebox.askyesno("確認", "選択した項目を削除しますか？"):
            return
        
        item = self.tree.item(selection[0])
        idx = int(item['text']) - 1
        
        if 0 <= idx < len(self.filtered_passwords):
            old_entry = self.filtered_passwords[idx]
            self.passwords.remove(old_entry)
            
            if self.save_data():
                messagebox.showinfo("成功", "データを削除しました")
                self.clear_entries()
                self.filter_data()
    
    def clear_entries(self):
        """入力フィールドのクリア"""
        for key, widget in self.entries.items():
            if isinstance(widget, ttk.Combobox):
                widget.set("")
            else:
                widget.delete(0, tk.END)
        
        # 任意項目のクリア
        for name_entry, value_entry in self.custom_fields:
            name_entry.delete(0, tk.END)
            value_entry.delete(0, tk.END)
    
    def export_csv(self):
        """CSVエクスポート"""
        file_path = filedialog.asksaveasfilename(
            defaultextension=".csv",
            filetypes=[("CSV files", "*.csv"), ("All files", "*.*")],
            initialfile=f"passwords_export_{datetime.now().strftime('%Y%m%d_%H%M%S')}.csv"
        )
        
        if not file_path:
            return
        
        try:
            with open(file_path, 'w', newline='', encoding='utf-8-sig') as f:
                if not self.passwords:
                    messagebox.showinfo("情報", "エクスポートするデータがありません")
                    return
                
                # すべてのフィールド名を収集
                fieldnames = ["大区分", "小区分", "種類", "アカウント/ID名", "パスワード", 
                             "セキュリティコード", "アドレス", "作成日時", "更新日時"]
                
                # カスタムフィールド名を収集
                custom_field_names = set()
                for pwd in self.passwords:
                    if "custom_fields" in pwd:
                        custom_field_names.update(pwd["custom_fields"].keys())
                
                fieldnames.extend(sorted(custom_field_names))
                
                writer = csv.DictWriter(f, fieldnames=fieldnames)
                writer.writeheader()
                
                # データの書き込み
                for pwd in self.passwords:
                    row = {
                        "大区分": pwd.get("大区分", ""),
                        "小区分": pwd.get("小区分", ""),
                        "種類": pwd.get("種類", ""),
                        "アカウント/ID名": pwd.get("アカウント/ID名", ""),
                        "パスワード": pwd.get("パスワード", ""),
                        "セキュリティコード": pwd.get("セキュリティコード", ""),
                        "アドレス": pwd.get("アドレス", ""),
                        "作成日時": pwd.get("作成日時", ""),
                        "更新日時": pwd.get("更新日時", "")
                    }
                    
                    # カスタムフィールドを追加
                    if "custom_fields" in pwd:
                        for field_name, field_value in pwd["custom_fields"].items():
                            row[field_name] = field_value
                    
                    writer.writerow(row)
            
            messagebox.showinfo("成功", f"CSVファイルをエクスポートしました:\n{file_path}")
        except Exception as e:
            messagebox.showerror("エラー", f"エクスポートに失敗しました: {str(e)}")

if __name__ == "__main__":
    root = tk.Tk()
    app = PasswordManager(root)
    root.mainloop()
